<?php
// load_data.php
header('Content-Type: application/json');
include 'db_connection.php';

// ==========================
//       پارامترها
// ==========================
$maxAttacks = 5;           // حداکثر تعداد حملات همزمان
$attackInterval = 8*3600;  // فاصله زمانی هر حمله (ثانیه)

// ==========================
//       تابع محاسبه حملات مجدد
// ==========================
/**
 * تابع اضافه کردن حملات مجدد به شهرهای آزاد شده
 * @param array $mapData داده مپ کاربر
 * @param array $mapConfig کانفیگ کامل مپ
 * @param int $maxAttacks حداکثر تعداد حملات همزمان
 * @param int $attackInterval فاصله زمانی هر حمله (ثانیه)
 * @return array مقداردهی شده مپ داده شامل حملات و next_attack_time
 */
function calculatePendingAttacks($mapData, $mapConfig, $maxAttacks, $attackInterval) {
    $now = time();
    $attacks = isset($mapData['attacks']) ? $mapData['attacks'] : [];
    $nextAttackTime = isset($mapData['next_attack_time']) ? (int)$mapData['next_attack_time'] : $now + $attackInterval;

    $mapLevel = isset($mapData['map_level']) ? (int)$mapData['map_level'] : 0;
    $progress = isset($mapData['progress']) ? (int)$mapData['progress'] : 0;

    // --- آماده سازی لیست شهرهای آزاد شده ---
    $unlockedCities = [];
    foreach ($mapConfig['regions'] as $region) {
        $rid = (int)$region['id'];
        $citiesCount = count($region['cities']);

        if ($rid < $mapLevel) {
            // تمام شهرهای سرزمین‌های کاملاً آزاد شده
            for ($cid=1; $cid<=$citiesCount; $cid++) {
                $unlockedCities[] = ['region'=>$rid, 'city'=>$cid];
            }
        } elseif ($rid == $mapLevel) {
            // شهرهای آزاد شده در آخرین سرزمین
            for ($cid=1; $cid<=$progress; $cid++) {
                $unlockedCities[] = ['region'=>$rid, 'city'=>$cid];
            }
        }
    }

    // --- مجموعه حملات فعلی برای چک سریع ---
    $attackedSet = [];
    foreach ($attacks as $a) {
        $attackedSet[] = $a['region'].'_'.$a['city'];
    }

    // --- بررسی تعداد حملات جدید بر اساس زمان ---
    if ($now >= $nextAttackTime) {
        $elapsed = $now - $nextAttackTime + $attackInterval;
        $intervalsPassed = floor($elapsed / $attackInterval);

        // تعداد حملاتی که باید اضافه شود (تا سقف maxAttacks)
        $slotsToAdd = min($intervalsPassed, $maxAttacks - count($attacks));

        // انتخاب شهرهای آزاد شده که در لیست حمله نیستند
        $eligible = [];
        foreach ($unlockedCities as $city) {
            $key = $city['region'].'_'.$city['city'];
            if (!in_array($key, $attackedSet)) $eligible[] = $city;
        }

        for ($i=0; $i<$slotsToAdd; $i++) {
            if (count($eligible) == 0) break;
            $idx = array_rand($eligible);
            $pick = $eligible[$idx];
            $attacks[] = $pick;
            $attackedSet[] = $pick['region'].'_'.$pick['city'];
            array_splice($eligible, $idx, 1);
        }

        // آپدیت زمان حمله بعدی
        $nextAttackTime += $intervalsPassed * $attackInterval;
    }

    // ذخیره مقادیر جدید در mapData
    $mapData['attacks'] = $attacks;
    $mapData['next_attack_time'] = $nextAttackTime;

    return $mapData;
}

// ==========================
//      دریافت user_id
// ==========================
$data = json_decode(file_get_contents("php://input"), true);
if (!$data || !isset($data['user_id'])) {
    http_response_code(400);
    echo json_encode(['status'=>'error','message'=>'invalid_input']);
    exit;
}
$user_id = (int)$data['user_id'];

// ==========================
//      خواندن اطلاعات کاربر
// ==========================
$stmt = $conn->prepare("SELECT player_level, map_data, assets, equipments FROM users WHERE user_id=?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
if (!$row = $result->fetch_assoc()) {
    echo json_encode(['status'=>'error','message'=>'user_not_found']);
    exit;
}

// decode map_data
$mapData = json_decode($row['map_data'], true);
if ($mapData === null) {
    $mapData = [
        'map_level' => 0,
        'progress' => 0,
        'attacks' => [],
        'next_attack_time' => time() + $attackInterval
    ];
}

// ==========================
//      لود map_config
// ==========================
$configPath = __DIR__ . '/map_config.json';
if (!file_exists($configPath)) {
    echo json_encode(['status'=>'error','message'=>'map_config_missing']);
    exit;
}
$mapConfig = json_decode(file_get_contents($configPath), true);

// ==========================
//      محاسبه حملات مجدد
// ==========================
$mapData = calculatePendingAttacks($mapData, $mapConfig, $maxAttacks, $attackInterval);

// ==========================
//      ذخیره دوباره در دیتابیس
// ==========================
$newMapJson = json_encode($mapData, JSON_UNESCAPED_UNICODE);
$stmt = $conn->prepare("UPDATE users SET map_data=? WHERE user_id=?");
$stmt->bind_param("si", $newMapJson, $user_id);
$stmt->execute();

// ==========================
//      ارسال خروجی به یونیتی
// ==========================
echo json_encode([
    'status'=>'success',
    'player_level' => (int)$row['player_level'],
    'map_data' => $mapData,
    'assets' => json_decode($row['assets'], true),
    'equipments' => json_decode($row['equipments'], true)
]);

$conn->close();
exit;
?>
