<?php
// update_progress.php
header('Content-Type: application/json');
include 'db_connection.php';

$input = json_decode(file_get_contents("php://input"), true);
if (!$input || !isset($input['user_id'])) {
    http_response_code(400);
    echo json_encode(['status'=>'error','message'=>'invalid_input']);
    exit;
}
$user_id = (int)$input['user_id'];

// load current map data
$stmt = $conn->prepare("SELECT map_level FROM users WHERE user_id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$res = $stmt->get_result();
if (!$row = $res->fetch_assoc()) {
    echo json_encode(['status'=>'error','message'=>'user_not_found']);
    exit;
}

$mapData = json_decode($row['map_level'], true);
if ($mapData === null) $mapData = ['map_level'=>0,'progress'=>0,'attacks'=>[], 'next_attack_time'=>time()+8*3600];

// load map_config for city counts
$mapConfig = json_decode(file_get_contents(__DIR__.'/map_config.json'), true);

$map_level = isset($mapData['map_level']) ? (int)$mapData['map_level'] : 0;
$progress = isset($mapData['progress']) ? (int)$mapData['progress'] : 0;
$attacks = isset($mapData['attacks']) ? $mapData['attacks'] : [];
$nextAttack = isset($mapData['next_attack_time']) ? (int)$mapData['next_attack_time'] : time() + 8*3600;

// determine next region id (the region player is currently progressing)
$nextRegionId = $map_level + 1;
$totalCities = 0;
foreach ($mapConfig['regions'] as $r) {
    if ($r['id'] == $nextRegionId) {
        $totalCities = count($r['cities']);
        break;
    }
}
if ($totalCities <= 0) {
    // no more regions -> player finished all regions
    echo json_encode(['status'=>'finished']);
    exit;
}

// increment progress by 1 (player just cleared the next city in order)
$progress += 1;

// remove attacked entry if it matches the city that was cleared (if it was under attack)
$clearedCityId = $progress; // since progress indicates number of cities cleared after increment; the city id just cleared is progress
// but careful: if city id base is 1 and we incremented from 0 -> cleared city 1
// remove from attacks if exists
$filtered = [];
foreach ($attacks as $a) {
    if (!($a['region'] == $nextRegionId && $a['city'] == $clearedCityId)) {
        $filtered[] = $a;
    }
}
$attacks = $filtered;

// if after increment progress == totalCities -> region fully cleared
if ($progress >= $totalCities) {
    $map_level += 1;
    $progress = 0;
    // note: unlocking next region progress starts at 0
}

// if attacks array size decreased (i.e., slot freed) and nextAttack wasn't set to future, set nextAttack = now + 8h
// We'll set nextAttack only when attacks decreased (slot freed) and if nextAttack < now (meaning not scheduled)
$now = time();
$oldCount = isset($row['map_level']) ? null : null; // not used
// simpler: always if slot freed then if nextAttack < now set nextAttack = now + 8h
// calculate old attacks count from previous mapData if possible
// For simplicity, we set nextAttack to current nextAttack if exists; if slot freed we ensure nextAttack at least now+8h
// Count slots freed:
$prevAttacksCount = isset($mapData['attacks']) ? count($mapData['attacks']) : 0;
$newAttacksCount = count($attacks);
if ($newAttacksCount < $prevAttacksCount) {
    if ($nextAttack <= $now) {
        $nextAttack = $now + 8*3600;
    }
}

// save mapData back
$mapData['map_level'] = $map_level;
$mapData['progress'] = $progress;
$mapData['attacks'] = $attacks;
$mapData['next_attack_time'] = $nextAttack;

$newJson = json_encode($mapData, JSON_UNESCAPED_UNICODE);
$stmt = $conn->prepare("UPDATE users SET map_level = ? WHERE user_id = ?");
$stmt->bind_param("si", $newJson, $user_id);
$stmt->execute();

echo json_encode(['status'=>'success','user_map'=>$mapData]);
$conn->close();
exit;
?>
