<?php
// get_user_map.php
header('Content-Type: application/json');
include 'db_connection.php';

// ورودی: { "user_id": 52 }
// خروجی: { "status":"success", "user_map": { ... } }

$input = json_decode(file_get_contents("php://input"), true);
if (!$input || !isset($input['user_id'])) {
    http_response_code(400);
    echo json_encode(['status'=>'error','message'=>'invalid_input']);
    exit;
}

$user_id = (int)$input['user_id'];

// 1) خواندن map_json فعلی از دیتابیس
$stmt = $conn->prepare("SELECT map_level FROM users WHERE user_id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$res = $stmt->get_result();
if (!$row = $res->fetch_assoc()) {
    echo json_encode(['status'=>'error','message'=>'user_not_found']);
    exit;
}

$mapJson = $row['map_level'];
if ($mapJson === null || $mapJson === '') {
    // اگر خالی بود، مقدار پیش‌فرض می‌سازیم
    $default = [
        'map_level' => 0,
        'progress' => 0,
        'attacks' => [],
        'next_attack_time' => time() + 8*3600 // شروع زمان بعدی 8 ساعت بعد
    ];
    $mapJson = json_encode($default);
}

// decode to array for manipulation
$mapData = json_decode($mapJson, true);
if ($mapData === null) {
    // در صورت JSON نامعتبر، recreate
    $mapData = [
        'map_level' => 0,
        'progress' => 0,
        'attacks' => [],
        'next_attack_time' => time() + 8*3600
    ];
}

// 2) Read map_config for valid regions & cities (we need sizes)
$configPath = __DIR__ . '/map_config.json';
if (!file_exists($configPath)) {
    echo json_encode(['status'=>'error','message'=>'map_config_missing']);
    exit;
}
$mapConfig = json_decode(file_get_contents($configPath), true);

// helper: get total cities in a region id
function getRegionCityCount($config, $regionId) {
    foreach ($config['regions'] as $r) {
        if ($r['id'] == $regionId) return count($r['cities']);
    }
    return 0;
}

// 3) Auto-generate attacks if needed (based on next_attack_time and max 5 attacks)
// Logic: while (now >= next_attack_time AND attacks_count < 5) -> add one random eligible city, and next_attack_time += 8h
$now = time();
$attacks = isset($mapData['attacks']) ? $mapData['attacks'] : [];
$nextAttack = isset($mapData['next_attack_time']) ? (int)$mapData['next_attack_time'] : $now + 8*3600;

// Get list of eligible cities (region > map_level OR region == map_level+1 and city index > progress)
// We must exclude already unlocked cities and already attacked cities.
$eligible = [];

// Build set of attacked keys for quick check
$attackedSet = [];
foreach ($attacks as $a) {
    $attackedSet[] = $a['region'].'_'.$a['city'];
}

// map_level current player's last fully cleared region
$map_level = isset($mapData['map_level']) ? (int)$mapData['map_level'] : 0;
$progress = isset($mapData['progress']) ? (int)$mapData['progress'] : 0;

// For each region in config, find eligible cities according to rules: only regions >= map_level+1 are valid to be attacked
foreach ($mapConfig['regions'] as $region) {
    $rid = (int)$region['id'];
    $cityCount = count($region['cities']);

    // We only allow attacks on regions that are unlocked or next ones?
    // According to your rule, attacks can happen on any previously freed or last in-progress region.
    // We'll allow attacks on any city that is not yet unlocked (i.e., in region > map_level OR in (region==map_level+1 and cityIndex > progress))
    if ($rid < $map_level + 1) continue; // skip fully cleared regions

    // startCity index that is considered "locked" (cities are 1-based)
    $startIndex = 1;
    if ($rid == $map_level + 1) {
        $startIndex = $progress + 1; // next locked city in the current progress region
    } else {
        $startIndex = 1;
    }

    for ($ci = $startIndex; $ci <= $cityCount; $ci++) {
        $key = $rid . '_' . $ci;
        if (in_array($key, $attackedSet)) continue; // already attacked
        // push eligible
        $eligible[] = ['region'=>$rid, 'city'=>$ci];
    }
}

// function to pick random eligible and add to attacks
function addRandomAttack(&$attacks, &$attackedSet, &$eligible) {
    if (count($eligible) == 0) return false;
    $idx = array_rand($eligible);
    $pick = $eligible[$idx];
    $attacks[] = ['region' => $pick['region'], 'city' => $pick['city']];
    $attackedSet[] = $pick['region'].'_'.$pick['city'];
    // remove picked from eligible
    array_splice($eligible, $idx, 1);
    return true;
}

// while loop to fill attack slots based on time
$maxAttacks = 5;
$changed = false;
while ($now >= $nextAttack && count($attacks) < $maxAttacks) {
    // try add
    if (!addRandomAttack($attacks, $attackedSet, $eligible)) break; // no eligible left
    $changed = true;
    // increment nextAttack by 8 hours
    $nextAttack += 8 * 3600;
}

// If attacks were removed/added or nextAttack changed, save back to DB
if ($changed || $nextAttack != (isset($mapData['next_attack_time']) ? (int)$mapData['next_attack_time'] : 0)) {
    $mapData['attacks'] = $attacks;
    $mapData['next_attack_time'] = $nextAttack;
    $newJson = json_encode($mapData, JSON_UNESCAPED_UNICODE);

    $stmt = $conn->prepare("UPDATE users SET map_level = ? WHERE user_id = ?");
    $stmt->bind_param("si", $newJson, $user_id);
    $stmt->execute();
}

// 4) return
echo json_encode(['status'=>'success','user_map' => $mapData, 'map_config' => $mapConfig]);
$conn->close();
exit;
?>
